//=============================================================================
// SoR_KeyMapAssignmentExtension_MZ.js
// SoR License (C) 2020 蒼竜, REQUIRED User Registration on Dragon Cave
// http://dragonflare.blue/dcave/license.php
// ---------------------------------------------------------------------------
// Latest version v1.01 (2021/03/03)
//=============================================================================
/*:ja
@plugindesc ＜キーボード入力対応拡張＞ v1.01
@author 蒼竜
@target MZ
@url http://dragonflare.blue.jp/dcave/
@help RPGツクールMZ製ゲームが、ゲーム処理中に受け取ることのできる
キーボードからの入力の種類を拡張します。

日本語キーボード(109キー)を想定して作成しているため、
USキーボードとの完全対応は保証いたしません。
細かいキー対応及び，各プラグインのキー指定プラグインパラメータでの
キー指定コードはpdfヘルプにまとめています。
@param --デバッグ・動作確認用--
@param DebuggingKey
@desc 確認目的でゲーム中に指定キーの入力を監視し、押下時にF8コンソールに出力する(無記入で処理しない)
@default 
@type string
@param ---拡張キー設定---
@param KeyA-Z
@desc 'true': キーボードのアルファベットを新規に受け取るようにする[既存機能上書き注意]
@default false
@type boolean
@param Key0-9
@desc 'true': キーボード数字(非テンキー)を新規に受け取るようにする
@default false
@type boolean
@param KeySigns
@desc 'true': キーボードの各種記号を新規に受け取るようにする
@default false
@type boolean
@param KeyNumeric
@desc 'true': テンキーを新規に受け取るようにする[既存機能上書き注意]
@default false
@type boolean
@param KeyNumericSigns
@desc 'true': テンキーパッドの各種記号・演算子(*や-など)を新規に受け取るようにする
@default false
@type boolean
@param KeySpecials
@desc 'true': キーボードの各種機能キーを新規に受け取るようにする[既存機能上書き注意]
@default false
@type boolean
@param KeyFunctions
@desc 'true': ファンクションキー(F1-F12)を新規に受け取るようにする[既存機能上書き注意]
@default false
@type boolean
@param ---重複キー定義---
@param DisableMappingOverwrite
@desc 'true': デフォルトで定義済みのキー設定を上書きしないように、キー入力受理を拡張する
@default false
@type boolean
*/
/*:
@plugindesc <Key Mapping Assignment Extention> v1.01
@author Soryu
@target MZ
@url http://dragonflare.blue.jp/dcave/index_e.php
@help This plugin extends input keys via keyboards
by assigning additional key mappings, which makes RPGMaker MZ games
to accept more key input patterns.

Expecting the key mapping of Japanese keyboards (109 keys),
complete consistency with US keyboards are not guaranteed.

Text codes associating with specific key inputs which must be
required by plugins that uses additional key inputs are 
presented in the pdf document.
@param --For Testing--
@param DebuggingKey
@desc Observing the designated key input to check the result in F8 console. (Set nothing to disable.)
@default 
@type string
@param --Key Extensions--
@param KeyA-Z
@desc 'true': Accept alphabet keys. [Beware overwriting some default settings.]
@default false
@type boolean
@param Key0-9
@desc 'true': Accept Number keys. (Not Numeric keypad)
@default false
@type boolean
@param KeySigns
@desc 'true': Accept some symbol keys on the keyboard.
@default false
@type boolean
@param KeyNumeric
@desc 'true': Accept Numeric keypad. [Beware overwriting some default settings.]
@default false
@type boolean
@param KeyNumericSigns
@desc 'true': Accept some symbol keys on Numeric keypad.
@default false
@type boolean
@param KeySpecials
@desc 'true': Accept some functional keys. [Beware overwriting some default settings.]
@default false
@type boolean
@param KeyFunctions
@desc 'true': Accept Function keys (F1-12). [Beware overwriting some default settings.]
@default false
@type boolean
@param ---Mapping Overwrite---
@param DisableMappingOverwrite
@desc 'true': Extend the key maps without overwriting the exisiting maps in default.
@default false
@type boolean
*/

(function() {
    const Param = PluginManager.parameters('SoR_KeyMapAssignmentExtension_MZ');

    const DebuggingKey = String(Param['DebuggingKey']);
    const IsAtoZ = Boolean(Param['KeyA-Z'] === 'true') || false;
    const Is0to9 = Boolean(Param['Key0-9'] === 'true') || false;
    const IsKeySigns = Boolean(Param['KeySigns'] === 'true') || false;
    const IsKeyNumeric = Boolean(Param['KeyNumeric'] === 'true') || false;
    const IsKeyNumericSigns = Boolean(Param['KeyNumericSigns'] === 'true') || false;
    const IsKeySpecials = Boolean(Param['KeySpecials'] === 'true') || false;
    const IsKeyFunctions = Boolean(Param['KeyFunctions'] === 'true') || false;
// v1.01
    const DisableMappingOverwrite = Boolean(Param['DisableMappingOverwrite'] === 'true') || false;

    if(Is0to9){
        for(let i=48;i<=57; i++){//0-9
            assignSingleMap(i,String.fromCharCode(i));
        }
    }

    if(IsAtoZ){
        for(let i=65;i<=90; i++){//A-Z
            assignSingleMap(i,String.fromCharCode(i));
        }
    }

    if(IsKeyNumeric){
        for(let i=96;i<=105; i++){//T0-T9            
            assignSingleMap(i,'T'+(i-96));
        }
    }

    if(IsKeyNumericSigns){
        //T_signs
        assignSingleMap(106,'T*');
        assignSingleMap(107,'T+');
        assignSingleMap(108,'T-');
        assignSingleMap(110,'T.');
        assignSingleMap(111,'T/');
    }

    if(IsKeySigns){
        //signs
        assignSingleMap(186,':*');
        assignSingleMap(187,';+');
        assignSingleMap(188,',<');
        assignSingleMap(189,'-=');
        assignSingleMap(190,'.>');
        assignSingleMap(191,'/?');
        assignSingleMap(192,'@`');
        assignSingleMap(219,'[{');
        assignSingleMap(220,'\\|');
        assignSingleMap(221,']}');
        assignSingleMap(222,'^~');
        assignSingleMap(226,'\\_');
    }

    if(IsKeySpecials){
        //others
        assignSingleMap(8,'backspace');
        assignSingleMap(18,'alt');
        assignSingleMap(20,'capslock');
        assignSingleMap(45,'insert');
        assignSingleMap(46,'delete');
        assignSingleMap(91,'win');
        assignSingleMap(93,'apps');
    }

    if(IsKeyFunctions){
        //functions
        for(let i=112;i<=123; i++){//T0-T9
            let flag = true;
            if(DisableMappingOverwrite) flag = !!!Input.keyMapper[i];
            if(flag) Input.keyMapper[i] = 'F'+(i-111);
        }
    }

    function assignSingleMap(i,code){
        let flag = true;
        if(DisableMappingOverwrite) flag = !!!Input.keyMapper[i];
        if(flag) Input.keyMapper[i] = code;
    }

    //for debugging
    if(DebuggingKey!=""){
        const SoR_KMAE_SB_update = Scene_Base.prototype.update;
        Scene_Base.prototype.update = function() {
            SoR_KMAE_SB_update.call(this);
            if(Input.isTriggered(DebuggingKey)) console.log("[Debugging: SoR_KeyMapAssignmentExtension_MZ] "+ DebuggingKey + " is Pressed!");
        }
    }

}());